{*******************************************************}
{                                                       }
{         Delphi VCL Extensions (RX)                    }
{                                                       }
{         Copyright (c) 1995, 1996 AO ROSNO             }
{                                                       }
{*******************************************************}

unit DBSecur;

interface

{$I RX.INC}

uses SysUtils, {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  Messages, Classes, Graphics, Controls, Forms, Dialogs, DB, DBTables,
  LoginDlg, ChPswDlg;

type

{ TDBSecurity }

  TUpdateCaption = (ucNoChange, ucAppTitle, ucFormCaption);
  TCheckUserEvent = function(UsersTable: TTable;
    const Password: string): Boolean of object;

  TDBSecurity = class(TComponent)
  private
    FActive: Boolean;
    FAttemptNumber: Integer;
    FDatabase: TDatabase;
    FUsersTableName: TFileName;
    FLoginNameField: PString;
    FLoggedUser: PString;
    FMaxPasswordLen: Integer;
    FAllowEmpty: Boolean;
    FUpdateCaption: TUpdateCaption;
    FIniFileName: PString;
    FSelectAlias: Boolean;
    FUseRegistry: Boolean;
    FLocked: Boolean;
    FUnlockDlgShowing: Boolean;
    FSaveOnRestore: TNotifyEvent;
    FOnCheckUser: TCheckUserEvent;
    FAfterLogin: TNotifyEvent;
    FBeforeLogin: TNotifyEvent;
    FOnChangePassword: TChangePasswordEvent;
    FOnUnlock: TCheckUnlockEvent;
    FOnIconDblClick: TNotifyEvent;
    procedure SetDatabase(Value: TDatabase);
    procedure SetUsersTableName(const Value: TFileName);
    function GetLoginNameField: string;
    procedure SetLoginNameField(const Value: string);
    function GetLoggedUser: string;
    procedure SetLoggedUser(const Value: string);
    function GetIniFileName: string;
    procedure SetIniFileName(const Value: string);
    function UnlockHook(var Message: TMessage): Boolean;
  protected
    procedure DoAfterLogin; dynamic;
    procedure DoBeforeLogin; dynamic;
    procedure DoIconDblCLick(Sender: TObject); dynamic;
    function DoCheckUser(UsersTable: TTable; const UserName,
      Password: string): Boolean; dynamic;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Login: Boolean;
    function ChangePassword: Boolean;
    procedure Lock;
    property LoggedUser: string read GetLoggedUser;
  published
    property Active: Boolean read FActive write FActive default True;
    property AllowEmptyPassword: Boolean read FAllowEmpty write FAllowEmpty default True;
    property AttemptNumber: Integer read FAttemptNumber write FAttemptNumber default 3;
    property Database: TDatabase read FDatabase write SetDatabase;
    property IniFileName: string read GetIniFileName write SetIniFileName;
    property LoginNameField: string read GetLoginNameField write SetLoginNameField;
    property MaxPasswordLen: Integer read FMaxPasswordLen write FMaxPasswordLen default 0;
    property UpdateCaption: TUpdateCaption read FUpdateCaption write FUpdateCaption default ucNoChange;
    property SelectAlias: Boolean read FSelectAlias write FSelectAlias default False;
{$IFDEF WIN32}
    property UseRegistry: Boolean read FUseRegistry write FUseRegistry default False;
{$ENDIF WIN32}
    property UsersTableName: TFileName read FUsersTableName write SetUsersTableName;
    property AfterLogin: TNotifyEvent read FAfterLogin write FAfterLogin;
    property BeforeLogin: TNotifyEvent read FBeforeLogin write FBeforeLogin;
    property OnCheckUser: TCheckUserEvent read FOnCheckUser write FOnCheckUser;
    property OnChangePassword: TChangePasswordEvent read FOnChangePassword
      write FOnChangePassword;
    property OnUnlock: TCheckUnlockEvent read FOnUnlock write FOnUnlock;
    property OnIconDblClick: TNotifyEvent read FOnIconDblClick write FOnIconDblClick;
  end;

implementation

uses AppUtils, VCLUtils;

{ TDBSecurity }

constructor TDBSecurity.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FIniFileName := NullStr;
  FActive := True;
  FAttemptNumber := 3;
  FAllowEmpty := True;
  FSelectAlias := False;
  FUseRegistry := False;
  FLoginNameField := NullStr;
  FLoggedUser := NullStr;
end;

destructor TDBSecurity.Destroy;
begin
  if FLocked then begin
    Application.UnhookMainWindow(UnlockHook);
    FLocked := False;
  end;
  DisposeStr(FLoggedUser);
  DisposeStr(FLoginNameField);
  DisposeStr(FIniFileName);
  inherited Destroy;
end;

procedure TDBSecurity.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (AComponent = Database) then Database := nil;
end;

procedure TDBSecurity.Loaded;
begin
  inherited Loaded;
  if not (csDesigning in ComponentState) and Active and
    (Database <> nil) then
  begin
    Database.LoginPrompt := True;
    if not Login then begin
      with Application do begin
        Terminate;
        if Handle <> 0 then ShowOwnedPopups(Handle, False);
      end;
      Halt(10);
    end;
  end;
end;

procedure TDBSecurity.SetDatabase(Value: TDatabase);
begin
  if FDatabase <> Value then begin
    FDatabase := Value;
{$IFDEF WIN32}
    if Value <> nil then Value.FreeNotification(Self);
{$ENDIF}
  end;
end;

procedure TDBSecurity.SetUsersTableName(const Value: TFileName);
begin
  if FUsersTableName <> Value then
    FUsersTableName := Value;
end;

function TDBSecurity.GetIniFileName: string;
begin
  Result := FIniFileName^;
  if (Result = '') and not (csDesigning in ComponentState) then begin
{$IFDEF WIN32}
    if UseRegistry then Result := GetDefaultIniRegKey
    else Result := GetDefaultIniName;
{$ELSE}
    Result := GetDefaultIniName;
{$ENDIF}
  end;
end;

procedure TDBSecurity.SetIniFileName(const Value: string);
begin
  AssignStr(FIniFileName, Value);
end;

function TDBSecurity.GetLoginNameField: string;
begin
  Result := FLoginNameField^;
end;

procedure TDBSecurity.SetLoginNameField(const Value: string);
begin
  AssignStr(FLoginNameField, Value);
end;

function TDBSecurity.GetLoggedUser: string;
begin
  Result := FLoggedUser^;
end;

procedure TDBSecurity.SetLoggedUser(const Value: string);
begin
  AssignStr(FLoggedUser, Value);
end;

procedure TDBSecurity.DoAfterLogin;
begin
  if Assigned(FAfterLogin) then FAfterLogin(Self);
end;

procedure TDBSecurity.DoBeforeLogin;
begin
  if Assigned(FBeforeLogin) then FBeforeLogin(Self);
end;

function TDBSecurity.DoCheckUser(UsersTable: TTable; const UserName,
  Password: string): Boolean;
var
  SaveLoggedUser: string;
begin
  if Assigned(FOnCheckUser) then begin
    SaveLoggedUser := LoggedUser;
    try
      SetLoggedUser(UserName);
      Result := FOnCheckUser(UsersTable, Password);
    finally
      SetLoggedUser(SaveLoggedUser);
    end;
  end
  else Result := True;
end;

procedure TDBSecurity.DoIconDblCLick(Sender: TObject);
begin
  if Assigned(FOnIconDblClick) then FOnIconDblClick(Self);
end;

function TDBSecurity.Login: Boolean;
var
  LoginName: string;
  F: TForm;
  IconClick: TNotifyEvent;
begin
  LoginName := EmptyStr;
  if Assigned(FOnIconDblClick) then IconClick := DoIconDblCLick
  else IconClick := nil;
  DoBeforeLogin;
  Result := LoginDialog(Database, AttemptNumber, UsersTableName,
    LoginNameField, MaxPasswordLen, DoCheckUser, IconClick, LoginName,
    IniFileName, FUseRegistry, FSelectAlias);
  if Result then begin
    SetLoggedUser(LoginName);
    F := Application.MainForm;
    if (F = nil) and (Owner is TForm) then F := Owner as TForm;
    if (F <> nil) and (LoginName <> '') then
      case UpdateCaption of
        ucAppTitle:
          F.Caption := Format('%s (%s)', [Application.Title, LoginName]);
        ucFormCaption:
          F.Caption := Format('%s (%s)', [F.Caption, LoginName]);
      end;
    DoAfterLogin;
  end;
end;

function TDBSecurity.ChangePassword: Boolean;
begin
  Result := ChangePasswordDialog(Database, AttemptNumber, UsersTableName,
    LoginNameField, LoggedUser, MaxPasswordLen, FAllowEmpty,
    OnChangePassword);
end;

procedure TDBSecurity.Lock;
begin
  FSaveOnRestore := Application.OnRestore;
  Application.Minimize;
  Application.HookMainWindow(UnlockHook);
  FLocked := True;
end;

function TDBSecurity.UnlockHook(var Message: TMessage): Boolean;

  function DoUnlock: Boolean;
  var
    Popup: HWnd;
  begin
    with Application do
      if IsWindowVisible(Handle) and IsWindowEnabled(Handle) then
{$IFDEF WIN32}
        SetForegroundWindow(Handle);
{$ELSE}
        BringWindowToTop(Handle);
{$ENDIF}
    if FUnlockDlgShowing then begin
      Popup := GetLastActivePopup(Application.Handle);
      if (Popup <> 0) and IsWindowVisible(Popup) and
        (WindowClassName(Popup) = TRxLoginForm.ClassName) then
      begin
{$IFDEF WIN32}
        SetForegroundWindow(Popup);
{$ELSE}
        BringWindowToTop(Popup);
{$ENDIF}
      end;
      Result := False;
      Exit;
    end;
    FUnlockDlgShowing := True;
    try
      Result := UnlockDialog(LoggedUser, OnUnlock, DoIconDblCLick);
    finally
      FUnlockDlgShowing := False;
    end;
    if Result then begin
      Application.UnhookMainWindow(UnlockHook);
      FLocked := False;
    end;
  end;

begin
  Result := False;
  if not FLocked then Exit;
  with Message do begin
    case Msg of
      WM_SHOWWINDOW:
        if Bool(WParam) then begin
          UnlockHook := not DoUnlock;
        end;
      WM_SYSCOMMAND:
        if (WParam and $FFF0 = SC_RESTORE) or
          (WParam and $FFF0 = SC_ZOOM) then
        begin
          UnlockHook := not DoUnlock;
        end;
    end;
  end;
end;

end.